# Copyright (c) 2014      Thomas Heller
# Copyright (c) 2007-2018 Hartmut Kaiser
# Copyright (c) 2011      Bryce Lelbach
#
# SPDX-License-Identifier: BSL-1.0
# Distributed under the Boost Software License, Version 1.0. (See accompanying
# file LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

cmake_policy(PUSH)

hpx_set_cmake_policy(CMP0054 NEW)
hpx_set_cmake_policy(CMP0060 NEW)

function(hpx_setup_target target)
  # retrieve arguments
  set(options EXPORT NOHPX_INIT INSTALL INSTALL_HEADERS INTERNAL_FLAGS NOLIBS PLUGIN
    NONAMEPREFIX NOTLLKEYWORD)
  set(one_value_args TYPE FOLDER NAME SOVERSION VERSION HPX_PREFIX HEADER_ROOT)
  set(multi_value_args DEPENDENCIES COMPONENT_DEPENDENCIES COMPILE_FLAGS LINK_FLAGS INSTALL_FLAGS)
  cmake_parse_arguments(target "${options}" "${one_value_args}" "${multi_value_args}" ${ARGN})

  if(NOT TARGET ${target})
    hpx_error("${target} does not represent a target")
  endif()

  # Figure out which type we want...
  if(target_TYPE)
    string(TOUPPER "${target_TYPE}" _type)
  else()
    get_target_property(type_prop ${target} TYPE)
    if(type_prop STREQUAL "STATIC_LIBRARY")
      set(_type "LIBRARY")
    endif()
    if(type_prop STREQUAL "MODULE_LIBRARY")
      set(_type "LIBRARY")
    endif()
    if(type_prop STREQUAL "SHARED_LIBRARY")
      set(_type "LIBRARY")
    endif()
    if(type_prop STREQUAL "EXECUTABLE")
      set(_type "EXECUTABLE")
    endif()
  endif()

  if(target_FOLDER)
    set_target_properties(${target} PROPERTIES FOLDER "${target_FOLDER}")
  endif()

  get_target_property(target_SOURCES ${target} SOURCES)

  # Manage files with .cu extension in case When Cuda Clang is used
  if(target_SOURCES AND HPX_WITH_CUDA_CLANG)
    foreach(source ${target_SOURCES})
      get_filename_component(extension ${source} EXT)
      if(${extension} STREQUAL ".cu")
        SET_SOURCE_FILES_PROPERTIES(${source} PROPERTIES
          COMPILE_FLAGS "${HPX_CUDA_CLANG_FLAGS}")
      endif()
    endforeach()
  endif()

  if(target_COMPILE_FLAGS)
    hpx_append_property(${target} COMPILE_FLAGS ${target_COMPILE_FLAGS})
  endif()

  if(target_LINK_FLAGS)
    hpx_append_property(${target} LINK_FLAGS ${target_LINK_FLAGS})
  endif()

  if(target_NAME)
    set(name "${target_NAME}")
  else()
    set(name "${target}")
  endif()

  if(target_NOTLLKEYWORD)
    set(__tll_private)
    set(__tll_public)
  else()
    set(__tll_private PRIVATE)
    set(__tll_public PUBLIC)
  endif()

  set(nohpxinit FALSE)
  if(target_NOHPX_INIT)
    set(nohpxinit TRUE)
  endif()

  set(target_STATIC_LINKING OFF)
  if(HPX_WITH_STATIC_LINKING)
    set(target_STATIC_LINKING ON)
  else()
    set(_hpx_library_type)
    if(TARGET hpx)
      get_target_property(_hpx_library_type hpx TYPE)
    endif()

    if("${_hpx_library_type}" STREQUAL "STATIC_LIBRARY")
      set(target_STATIC_LINKING ON)
    endif()
  endif()

  if(HPX_INCLUDE_DIRS)
    set_property(TARGET ${target} APPEND
      PROPERTY INCLUDE_DIRECTORIES
      "${HPX_INCLUDE_DIRS}"
    )
  endif()

  if("${_type}" STREQUAL "EXECUTABLE")
    if(target_HPX_PREFIX)
      set(_prefix ${target_HPX_PREFIX})
    else()
      set(_prefix ${HPX_PREFIX})
    endif()

    if(MSVC)
      string(REPLACE ";" ":" _prefix "${_prefix}")
    endif()

    set_property(TARGET ${target} APPEND
                 PROPERTY COMPILE_DEFINITIONS
                 "HPX_APPLICATION_NAME=${name}"
                 "HPX_APPLICATION_STRING=\"${name}\""
                 "HPX_PREFIX=\"${_prefix}\""
                 "HPX_APPLICATION_EXPORTS")
  endif()

  if("${_type}" STREQUAL "LIBRARY")
    set(nohpxinit FALSE)
    if(DEFINED HPX_LIBRARY_VERSION AND DEFINED HPX_SOVERSION)
      # set properties of generated shared library
      set_target_properties(${target}
        PROPERTIES
        VERSION ${HPX_LIBRARY_VERSION}
        SOVERSION ${HPX_SOVERSION})
    endif()
    if(NOT target_NONAMEPREFIX)
      hpx_set_lib_name(${target} ${name})
    endif()
    set_target_properties(${target}
      PROPERTIES
      # create *nix style library versions + symbolic links
      # allow creating static and shared libs without conflicts
      CLEAN_DIRECT_OUTPUT 1
      OUTPUT_NAME ${name})
    if(target_PLUGIN)
      set(plugin_name "HPX_PLUGIN_NAME=hpx_${name}")
    endif()
    set(nohpxinit TRUE)

    set_property(TARGET ${target} APPEND
                 PROPERTY COMPILE_DEFINITIONS
                 "HPX_LIBRARY_EXPORTS"
                 ${plugin_name})
  endif()

  if("${_type}" STREQUAL "COMPONENT")
    set(nohpxinit FALSE)
    if(DEFINED HPX_LIBRARY_VERSION AND DEFINED HPX_SOVERSION)
    # set properties of generated shared library
      set_target_properties(${target}
        PROPERTIES
        VERSION ${HPX_LIBRARY_VERSION}
        SOVERSION ${HPX_SOVERSION})
    endif()
    if(NOT target_NONAMEPREFIX)
      hpx_set_lib_name(${target} ${name})
    endif()
    set_target_properties(${target}
      PROPERTIES
      # create *nix style library versions + symbolic links
      # allow creating static and shared libs without conflicts
      CLEAN_DIRECT_OUTPUT 1
      OUTPUT_NAME ${name})
    set(nohpxinit TRUE)

    set_property(TARGET ${target} APPEND
                 PROPERTY COMPILE_DEFINITIONS
                 "HPX_COMPONENT_NAME=hpx_${name}"
                 "HPX_COMPONENT_STRING=\"hpx_${name}\""
                 "HPX_COMPONENT_EXPORTS")
  endif()

  # We force the -DDEBUG and -D_DEBUG defines in debug mode to avoid
  # ABI differences
  # if hpx is an imported target, get the config debug/release
  set(HPX_IMPORT_CONFIG "NOTFOUND")
  if (TARGET "hpx")
    get_target_property(HPX_IMPORT_CONFIG "hpx" IMPORTED_CONFIGURATIONS)
  endif()
  if(HPX_IMPORT_CONFIG MATCHES NOTFOUND)
    # we are building HPX not importing, so we should use the $<CONFIG:variable
    set(_USE_CONFIG 1)
  else()
    # hpx is an imported target, so set HPX_DEBUG based on build config of hpx library
    set(_USE_CONFIG 0)
  endif()

  if(NOT target_NOLIBS)
    if(HPX_WITH_DYNAMIC_HPX_MAIN AND ("${_type}" STREQUAL "EXECUTABLE") AND
       (("${CMAKE_SYSTEM_NAME}" STREQUAL "Linux") OR (APPLE)))
      set(hpx_libs hpx_wrap)
    endif()
    set(hpx_libs ${hpx_libs} hpx)
    if(NOT target_STATIC_LINKING)
      set(hpx_libs ${hpx_libs})
      if(NOT nohpxinit)
        set(hpx_libs hpx_init ${hpx_libs})
      endif()
    endif()
    hpx_handle_component_dependencies(target_COMPONENT_DEPENDENCIES)
    set(hpx_libs ${hpx_libs} ${target_COMPONENT_DEPENDENCIES})
    if("${CMAKE_CXX_COMPILER_ID}" STREQUAL "Intel")
      set(hpx_libs ${hpx_libs} imf svml irng intlc)
    endif()
  else()
    target_compile_options(${target} ${__tll_public} ${CXX_FLAG})
  endif()

  target_link_libraries(${target} ${__tll_public} ${hpx_libs} ${target_DEPENDENCIES})

  if(target_INTERNAL_FLAGS AND TARGET hpx_internal_flags)
    target_link_libraries(${target} ${__tll_private} hpx_internal_flags)
  endif()

  get_target_property(target_EXCLUDE_FROM_ALL ${target} EXCLUDE_FROM_ALL)

  if(target_EXPORT AND NOT target_EXCLUDE_FROM_ALL)
    hpx_export_targets(${target})
    set(install_export EXPORT HPXTargets)
  endif()

  if(target_INSTALL AND NOT target_EXCLUDE_FROM_ALL)
    install(TARGETS ${target}
      ${install_export}
      ${target_INSTALL_FLAGS}
    )
    if(target_INSTALL_HEADERS AND (NOT target_HEADER_ROOT STREQUAL ""))
      install(
        DIRECTORY "${target_HEADER_ROOT}/"
        DESTINATION "${CMAKE_INSTALL_INCLUDEDIR}"
        COMPONENT ${name})
    endif()
  endif()
endfunction()

cmake_policy(POP)
